<?php
/**
 * Plugin Name: Gorilla Desk Implementation
 * Plugin URI: https://bolor.me
 * Description: Integrates Gorilla Desk portal with WordPress. Author: Bolorerdene Bundgaa.
 * Version: 1.0.0
 * Author: Bolorerdene Bundgaa
 * Author URI: https://bolor.me
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: gorilla-desk
 * Domain Path: /languages
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('GORILLA_DESK_VERSION', '1.0.0');
define('GORILLA_DESK_PLUGIN_URL', plugin_dir_url(__FILE__));
define('GORILLA_DESK_PLUGIN_PATH', plugin_dir_path(__FILE__));

class GorillaDesk {
    
    public function __construct() {
        add_action('init', array($this, 'init'));
    }
    
    public function init() {
        // Add admin menu if user is admin
        if (is_admin()) {
            add_action('admin_menu', array($this, 'add_admin_menu'));
            add_action('admin_init', array($this, 'register_settings'));
        }
        
        // Add the script to frontend
        add_action('wp_footer', array($this, 'add_gorilla_script'));
        
        // Add settings link to plugin page
        add_filter('plugin_action_links_' . plugin_basename(__FILE__), array($this, 'add_settings_link'));
    }
    
    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        add_options_page(
            'Gorilla Desk Settings',
            'Gorilla Desk',
            'manage_options',
            'gorilla-desk',
            array($this, 'admin_page')
        );
    }
    
    /**
     * Register plugin settings
     */
    public function register_settings() {
        register_setting('gorilla_desk_settings', 'gorilla_desk_account_id');
        register_setting('gorilla_desk_settings', 'gorilla_desk_enabled');
        register_setting('gorilla_desk_settings', 'gorilla_desk_chatbot_enabled');
    }
    
    /**
     * Admin settings page
     */
    public function admin_page() {
        ?>
        <div class="wrap">
            <h1>Gorilla Desk Settings</h1>
            <form method="post" action="options.php">
                <?php
                settings_fields('gorilla_desk_settings');
                do_settings_sections('gorilla_desk_settings');
                ?>
                <table class="form-table">
                    <tr>
                        <th scope="row">Enable Gorilla Desk</th>
                        <td>
                            <input type="checkbox" name="gorilla_desk_enabled" value="1" <?php checked(1, get_option('gorilla_desk_enabled', 1)); ?> />
                            <label>Enable Gorilla Desk integration</label>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Account ID</th>
                        <td>
                            <input type="text" name="gorilla_desk_account_id" value="<?php echo esc_attr(get_option('gorilla_desk_account_id', '')); ?>" class="regular-text" required />
                            <p class="description">Your Gorilla Desk Account ID (required)</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Enable Chatbot</th>
                        <td>
                            <input type="checkbox" name="gorilla_desk_chatbot_enabled" value="1" <?php checked(1, get_option('gorilla_desk_chatbot_enabled', 0)); ?> />
                            <label>Enable Gorilla Desk chatbot</label>
                        </td>
                    </tr>
                </table>
                <?php submit_button(); ?>
            </form>
            
            <hr>
            <h2>About</h2>
            <p>Author: Bolorerdene Bundgaa</p>
            <p>Version: <?php echo GORILLA_DESK_VERSION; ?></p>
        </div>
        <?php
    }
    
    /**
     * Add settings link to plugin actions
     */
    public function add_settings_link($links) {
        $settings_link = '<a href="options-general.php?page=gorilla-desk">Settings</a>';
        array_unshift($links, $settings_link);
        return $links;
    }
    
    /**
     * Add Gorilla Desk script to footer
     */
    public function add_gorilla_script() {
        // Check if plugin is enabled
        if (!get_option('gorilla_desk_enabled', 1)) {
            return;
        }
        
        $account_id = get_option('gorilla_desk_account_id', '');
        $chatbot_enabled = get_option('gorilla_desk_chatbot_enabled', 0) ? 'true' : 'false';
        
        if (empty($account_id)) {
            return;
        }
        ?>
        <script type="text/javascript">
            var _gorilla = _gorilla || {};
            _gorilla.account_id = '<?php echo esc_js($account_id); ?>';
            _gorilla.loaded = false;
            _gorilla.chatbot = <?php echo $chatbot_enabled; ?>;
            
            var _gorillaInitPortal = function () {
                _gorilla.loaded = true;
                var a = document.createElement('script');
                a.type = 'text/javascript';
                a.async = true;
                a.defer = true;
                a.src = 'https://app.gorilladesk.com/js/portal/portal.js';
                var b = document.getElementsByTagName('script')[0];
                b.parentNode.insertBefore(a, b);
            };
            
            if (window.addEventListener) {
                window.addEventListener('load', _gorillaInitPortal, false);
            } else if (window.attachEvent) {
                window.attachEvent('onload', _gorillaInitPortal);
            } else {
                window.onload = _gorillaInitPortal;
            }
        </script>
        <?php
    }
    
    /**
     * Plugin activation hook
     */
    public static function activate() {
        // Set default options
        add_option('gorilla_desk_enabled', 1);
        add_option('gorilla_desk_account_id', '');
        add_option('gorilla_desk_chatbot_enabled', 0);
    }
    
    /**
     * Plugin deactivation hook
     */
    public static function deactivate() {
        // Clean up if needed
    }
    
    /**
     * Plugin uninstall hook
     */
    public static function uninstall() {
        // Remove options
        delete_option('gorilla_desk_enabled');
        delete_option('gorilla_desk_account_id');
        delete_option('gorilla_desk_chatbot_enabled');
    }
}

// Initialize the plugin
new GorillaDesk();

// Register activation/deactivation hooks
register_activation_hook(__FILE__, array('GorillaDesk', 'activate'));
register_deactivation_hook(__FILE__, array('GorillaDesk', 'deactivate'));
register_uninstall_hook(__FILE__, array('GorillaDesk', 'uninstall'));

?>
